# Fix Genre Input Limitation - Plan

## Problem Statement
When editing or creating a movie in the admin panel at `/admin/movie/1`, the genre input field is limited to a maximum of 5 genres. Users should be able to select unlimited genres.

## Root Cause Analysis

### Issue Location
The limitation is enforced in the JavaScript Selectize initialization in the movie form.

**File:** `resources/views/admin/post/form.blade.php`
**Line:** 216

```javascript
$('.selectpicker').selectize({maxItems: 5, create: false})
```

The `maxItems: 5` parameter restricts users to selecting only 5 genres.

### Backend Verification
The backend code in [`app/Http/Controllers/Admin/MovieController.php`](app/Http/Controllers/Admin/MovieController.php:160) properly handles unlimited genres:

- **Store method (line 160):** `$model->genres()->sync($request->genre);`
- **Update method (line 328):** `$model->genres()->sync($request->genre);`

The backend uses `sync()` which accepts an array of genre IDs with no inherent limit.

### Frontend Form
The genre input field is located in [`resources/views/admin/post/partials/overview.blade.php`](resources/views/admin/post/partials/overview.blade.php:26-36):

```blade
<div class="mb-5">
    <x-form.label for="category" :value="__('Genre')"/>
    <div class="relative flex w-full">
        <select name="genre[]" placeholder="Choose" class="selectpicker" multiple id="genre" x-ref="categorySelect" required>
            <option value="">Choose</option>
            @foreach($genres as $genre)
                <option data-text="{{$genre->title}}" value="{{$genre->id}}"
                        @if(isset($listing->id) AND $listing->genres()->find($genre->id)) selected @endif>{{$genre->title}}</option>
            @endforeach
        </select>
    </div>
</div>
```

The HTML select element has the `multiple` attribute, which allows multiple selections. The limitation is purely client-side via Selectize.

## Solution

### Option 1: Remove the Limitation (Recommended)
Completely remove the `maxItems` parameter to allow unlimited genre selection.

**Change in:** `resources/views/admin/post/form.blade.php` (line 216)

**From:**
```javascript
$('.selectpicker').selectize({maxItems: 5, create: false})
```

**To:**
```javascript
$('.selectpicker').selectize({create: false})
```

### Option 2: Increase the Limit
If you want to keep some limit but increase it, modify the `maxItems` value.

**Example (limit to 10):**
```javascript
$('.selectpicker').selectize({maxItems: 10, create: false})
```

## Implementation Steps

1. **Backup the file** (optional but recommended)
   - Copy `resources/views/admin/post/form.blade.php`

2. **Edit the file**
   - Open `resources/views/admin/post/form.blade.php`
   - Navigate to line 216
   - Remove the `maxItems: 5,` parameter from the selectize initialization

3. **Test the changes**
   - Navigate to `/admin/movie/create` or `/admin/movie/1/edit`
   - Attempt to select more than 5 genres
   - Verify that all selected genres are saved properly

4. **Clear browser cache** (if needed)
   - The JavaScript change may require a hard refresh (Ctrl+Shift+R or Cmd+Shift+R)

## Impact Analysis

### Affected Areas
- Movie creation form
- Movie editing form
- Any other admin forms using the `.selectpicker` class

### Potential Side Effects
- None expected. The backend already supports unlimited genres
- The change only affects the client-side validation
- Other admin forms using `.selectpicker` will also have unlimited selections (this may be desirable)

### Database Impact
- No database changes required
- The `post_genres` pivot table already supports unlimited relationships

## Testing Checklist

- [ ] Create a new movie and select 6+ genres
- [ ] Verify all selected genres are saved
- [ ] Edit an existing movie and add more than 5 genres
- [ ] Verify the changes persist after saving
- [ ] Check that the genre display on the frontend shows all genres
- [ ] Test with TV shows (if they use the same form)

## Alternative Considerations

If you want to apply different limits to different select inputs, you could:

1. Use different CSS classes for different select inputs
2. Initialize Selectize separately for each input with different configurations

Example:
```javascript
$('.genre-picker').selectize({create: false})  // Unlimited genres
$('.tag-picker').selectize({maxItems: 10, create: false})  // Limited tags
```

This would require updating the HTML classes in the form partials.

## Conclusion

The fix is straightforward and low-risk. Simply removing or increasing the `maxItems` parameter in the Selectize initialization will allow users to select unlimited genres for movies and TV shows.
