<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\LandingSection;
use Cviebrock\EloquentSluggable\Services\SlugService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;

class LandingSectionController extends Controller
{
    public function index(Request $request)
    {
        $config = [
            'title' => __('Landing Sections'),
            'nav' => 'management',
            'route' => 'landing-section',
            'create' => true,
        ];

        // Filter
        $search = $request->input('q');
        $searchBy = in_array($request->input('search_by'), ['title']) ? $request->input('search_by') : 'title';
        $sort = in_array($request->sorting, ['asc', 'desc']) ? $request->sorting : 'desc';
        $perPage = config('attr.page_limit');

        // Query
        $listings = LandingSection::when($search, function ($query) use ($search) {
            return $query->where('title', 'like', '%' . $search . '%');
        })->with('genres')->orderBy('sortable', 'asc')->orderBy('id', 'desc')->paginate($perPage)->appends(['q' => $search, 'sort' => $sort]);

        return view('admin.landing-section.index', compact('config', 'listings'));
    }

    public function create()
    {
        $config = [
            'title' => __('Landing Section'),
            'nav' => 'management',
            'route' => 'landing-section',
        ];

        $genres = \App\Models\Genre::orderBy('title', 'asc')->get();

        return view('admin.landing-section.form', compact('config', 'genres'));
    }

    public function store(Request $request)
    {
        $this->validate($request, [
            'title' => 'required|string|max:255',
            'genres' => 'required|array|min:1',
            'genres.*' => 'required|exists:genres,id',
            'limit' => 'required|integer|min:1|max:50',
            'status' => 'required|in:active,inactive',
            'sortable' => 'integer',
        ]);

        $model = new LandingSection();

        $model->title = $request->input('title');
        $model->slug = SlugService::createSlug(LandingSection::class, 'slug', $request->input('title'));
        $model->limit = $request->input('limit');
        $model->status = $request->input('status');
        $model->sortable = $request->input('sortable', 0);
        $model->save();

        $model->genres()->sync($request->input('genres'));

        Cache::flush();
        return redirect()->route('admin.landing-section.index')->with('success', __(':title created', ['title' => $request->input('title')]));
    }

    public function edit($id)
    {
        $config = [
            'title' => __('Landing Section'),
            'nav' => 'management',
            'route' => 'landing-section',
        ];

        $listing = LandingSection::where('id', $id)->firstOrFail() ?? abort(404);
        $genres = \App\Models\Genre::orderBy('title', 'asc')->get();

        return view('admin.landing-section.form', compact('config', 'listing', 'genres'));
    }

    public function update(Request $request, $id)
    {
        $this->validate($request, [
            'title' => 'required|string|max:255',
            'genres' => 'required|array|min:1',
            'genres.*' => 'required|exists:genres,id',
            'limit' => 'required|integer|min:1|max:50',
            'status' => 'required|in:active,inactive',
            'sortable' => 'integer',
        ]);

        $model = LandingSection::findOrFail($id);
        $model->title = $request->input('title');
        if ($model->slug != $request->slug) {
            $model->slug = SlugService::createSlug(LandingSection::class, 'slug', $request->input('title'));
        }
        $model->limit = $request->input('limit');
        $model->status = $request->input('status');
        $model->sortable = $request->input('sortable', 0);
        $model->save();

        $model->genres()->sync($request->input('genres'));

        Cache::flush();
        return redirect()->route('admin.landing-section.edit', $model->id)->with('success', __(':title updated', ['title' => $request->input('title')]));
    }

    public function destroy($id)
    {
        LandingSection::find($id)->delete();

        return redirect()->back()->with('success', __('Deleted'));
    }
}
